﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Configurations;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Models.DentalEncounter;
    using Newtonsoft.Json;
    using Newtonsoft.Json.Serialization;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library;
    using Shared.Library.Enums;
    using Shared.UserModels;
    using Shared.UserModels.Filters;
    using Utilities;

    using Hims.Api.Helper;

    /// <summary>
    /// The dental encounter controller.
    /// </summary>
    [Authorize]
    [Route("api/dental-encounter")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DentalEncounterController : BaseController
    {

        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IEncounterService encounterServices;

        /// <summary>
        /// The push notification helper.
        /// </summary>
        private readonly IPushNotificationHelper pushNotificationHelper;

        /// <summary>
        /// The account session services.
        /// </summary>
        private readonly IAccountSessionService accountSessionServices;

        /// <summary>
        /// The dental encounter service.
        /// </summary>
        private readonly IDentalEncounterService dentalEncounterServices;

        /// <summary>
        /// the aes helper
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IInternalMedicineService internalMedicineService;

        /// <summary>
        /// The appointments services.
        /// </summary>
        private readonly IAppointmentService appointmentsServices;

        /// <summary>
        /// The account service.
        /// </summary>
        private readonly IAccountService accountService;

        /// <summary>
        /// The amazon s3 helper.
        /// </summary>
        private readonly IDocumentHelper documentHelper;

        /// <summary>
        /// The configuration.
        /// </summary>
        private readonly IAmazonS3Configuration configuration;

        /// <inheritdoc />
        public DentalEncounterController(
            IDentalEncounterService dentalEncounterServices,
            IDocumentHelper documentHelper,
            IAccountService accountService,
            IAmazonS3Configuration configuration,
            IAESHelper aesHelper,
            IEncounterService encounterServices,
            IPushNotificationHelper pushNotificationHelper,
            IAccountSessionService accountSessionServices,
            IInternalMedicineService internalMedicineService,
            IAppointmentService appointmentsServices)
        {
            this.internalMedicineService = internalMedicineService;
            this.dentalEncounterServices = dentalEncounterServices;
            this.aesHelper = aesHelper;
            this.configuration = configuration;
            this.documentHelper = documentHelper;
            this.accountService = accountService;
            this.appointmentsServices = appointmentsServices;
            this.encounterServices = encounterServices;
            this.pushNotificationHelper = pushNotificationHelper;
            this.accountSessionServices = accountSessionServices;
        }

        /// <summary>
        /// The find async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find")]
        [ProducesResponseType(typeof(DentalEncounterModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindAsync([FromBody] DentalEncounterFilterModel model)
        {
            model = (DentalEncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var dentalEncounter = await this.dentalEncounterServices.FindAsync(appointmentId);
            var providerId = !string.IsNullOrEmpty(model.EncryptedProviderId) ? Convert.ToInt32(this.aesHelper.Decode(model.EncryptedProviderId)) : 0;
            var patientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            var appointments = await this.appointmentsServices.FindByPatientAsync(patientId, providerId);
            var appointmentList = appointments.ToList();
            foreach (var item in appointmentList)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
                item.EncryptedPatientId = model.EncryptedPatientId;
                item.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd")).Add(item.AppointmentTime).ToString("hh:mm tt");
            }

            var oldAppointment = new PreviousAppointmentModel();
            if (dentalEncounter == null || dentalEncounter.DentalEncounterId == 0)
            {
                oldAppointment = await this.appointmentsServices.FindPreviousAppointmentAsync(appointmentId);
                if (oldAppointment != null)
                {
                    oldAppointment.EncryptedAppointmentId = this.aesHelper.Encode(oldAppointment.AppointmentId.ToString());
                }
            }

            return this.Success(new { Dashboard = dentalEncounter, Appointments = appointmentList, OldAppointment = oldAppointment });
        }

        /// <summary>
        /// The find async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find-dental-encounter")]
        [ProducesResponseType(typeof(DentalEncounterModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindDentalEncounterAsync([FromBody] DentalEncounterFilterModel model)
        {
            model = (DentalEncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var dentalEncounter = await this.dentalEncounterServices.FindAsync(appointmentId);

            return this.Success(dentalEncounter);
        }

        /// <summary>
        /// The find dashboard async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("find-dashboard")]
        [ProducesResponseType(typeof(DentalEncounterResource), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindDashboardAsync([FromBody] DentalEncounterFilterModel model)
        {
            model = (DentalEncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var response = await this.dentalEncounterServices.FetchAsync(appointmentId, model.Type);

            return this.Success(response);
        }

        /// <summary>
        /// The find prescription async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("find-prescription")]
        [ProducesResponseType(typeof(DentalPrescriptionModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindPrescriptionAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var internalMedicinePrescription = await this.dentalEncounterServices.FindPrescriptionAsync(appointmentId);
            if (internalMedicinePrescription == null || internalMedicinePrescription.DentalEncounterId == 0)
            {
                return this.BadRequest("Sorry! We don't have a encounter in the system.");
            }

            internalMedicinePrescription.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd")).Add(internalMedicinePrescription.AppointmentTime).ToString("hh:mm tt");
            return this.Success(internalMedicinePrescription);
        }

        /// <summary>
        /// The modify async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("modify")]
        [ProducesResponseType(typeof(int), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyAsync([FromBody] DentalEncounterModifyModel model)
        {
            model = (DentalEncounterModifyModel)EmptyFilter.Handler(model);
            var encounterId = model.DentalEncounterId == 0 
                                  ? await this.dentalEncounterServices.AddAltAsync(model) 
                                  : await this.dentalEncounterServices.UpdateAltAsync(model);
            switch (encounterId.Response)
            {
                case 0:
                    return this.ServerError();
                default:
                    var basicDetails = await this.encounterServices.GetBasicAppointmentDetails(model.AppointmentId, model.IsAdmission);
                    await this.appointmentsServices.UpdateEncounterTypeAsync(model.AppointmentId, (int)EncounterTypes.DentalEncounter, model.IsAdmission);

                    if (encounterId.Response > 1)
                    {
                        if (encounterId.Status == 1)
                        {
                            await NotificationHelper.Notification(
                            basicDetails.PatientId,
                            Roles.Patient,
                            model.DentalEncounterId == 0 ? NotificationIntimate.FullTranscriptionAdded : NotificationIntimate.FullTranscriptionUpdated,
                            this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                            this.accountSessionServices,
                            this.pushNotificationHelper);
                        }

                        if (encounterId.Status == 2)
                        {
                            await NotificationHelper.Notification(
                            basicDetails.PatientId,
                            Roles.Patient,
                            model.DentalEncounterId == 0 ? NotificationIntimate.PrescriptionAdded : NotificationIntimate.PrescriptionUpdated,
                            this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                            this.accountSessionServices,
                            this.pushNotificationHelper);
                        }
                    }
                    
                    return this.Success(encounterId.Response);
            }
        }

        /// <summary>
        /// The save encounter images.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("save-clinical-findings")]
        public async Task<ActionResult> SaveEncounterImages([FromBody] DentalEncounterImages model)
        {
            model = (DentalEncounterImages)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var getPatientAccount = await this.internalMedicineService.GetAccountBasedOnAppointmentId(appointmentId, model.IsAdmission);
            var guid = await this.accountService.FindGuidAsync(getPatientAccount.PatientId, Roles.Patient);

            foreach (var image in model.Images)
            {
                if (image.Base64String.Contains(this.configuration.BucketURL) || image.Base64String.Contains("assets"))
                {
                    continue;
                }

                string appointment;

                try
                {
                    appointment = getPatientAccount.AppointmentDate.Add(getPatientAccount.AppointmentTime)
                        .ToString("yyyy-MM-dd_hh:mm_tt");
                }
                catch
                {
                    appointment = getPatientAccount.AppointmentNo;
                }

                var folderName = $@"dentalEncounter/{appointment}/ClinicalFindings/{model.Type}";
                var url = await this.documentHelper.UploadEncounterImagesAsync(image.Base64String, guid, folderName, image.Type);
                image.Base64String = url;
            }

            var finalModel = new DentalEncounterModifyModel
            {
                AppointmentId = appointmentId,
                DentalEncounterId = model.DentalEncounterId ?? 0,
                JsonString = JsonConvert.SerializeObject(
                    model,
                    new JsonSerializerSettings { ContractResolver = new CamelCasePropertyNamesContractResolver() }),
                Type = model.Type,
                ModifiedBy = model.ModifiedBy
            };
            return await this.ModifyAsync(finalModel);
        }

        /// <summary>
        /// The find full transcript async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("find-full-transcript")]
        [ProducesResponseType(typeof(DentalFullTranscriptModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindFullTranscriptAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var fullTranscript = await this.dentalEncounterServices.FindFullTranscriptAsync(appointmentId);
            if (fullTranscript == null || fullTranscript.DentalEncounterId == 0)
            {
                return this.BadRequest("Sorry! We don't have a dental encounter in the system.");
            }

            fullTranscript.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd")).Add(fullTranscript.AppointmentTime).ToString("hh:mm tt");
            return this.Success(fullTranscript);
        }
    }
}
